import { useParams, Link } from "wouter";
import { Button } from "@/components/ui/button";
import { Card, CardContent, CardDescription, CardHeader, CardTitle } from "@/components/ui/card";
import { trpc } from "@/lib/trpc";
import { APP_LOGO } from "@/const";
import { Loader2, CheckCircle2, Clock, Download } from "lucide-react";
import ChatWidget from "@/components/ChatWidget";

export default function PaymentResult() {
  const { id } = useParams();
  const requestId = id ? parseInt(id) : 0;
  const { data: request, isLoading } = trpc.request.getById.useQuery(
    { id: requestId },
    { enabled: !!requestId && requestId > 0, refetchInterval: 5000 }
  );

  const getStatusText = (status: string) => {
    switch (status) {
      case "pending_payment":
        return "في انتظار الدفع";
      case "payment_submitted":
        return "تم إرسال إثبات الدفع - جاري المراجعة";
      case "paid":
        return "تم الدفع بنجاح - جاري إعداد التقرير";
      case "completed":
        return "تم الاكتمال";
      case "refunded":
        return "تم الاسترداد";
      default:
        return status;
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case "completed":
        return <CheckCircle2 className="h-16 w-16 text-green-600" />;
      case "paid":
      case "payment_submitted":
        return <Clock className="h-16 w-16 text-cyan-600 animate-pulse" />;
      default:
        return <Clock className="h-16 w-16 text-gray-400" />;
    }
  };

  return (
    <div className="min-h-screen bg-gradient-to-br from-purple-50 via-white to-cyan-50">
      {/* Header */}
      <header className="bg-white shadow-sm sticky top-0 z-50">
        <div className="container mx-auto px-4 py-4 flex justify-between items-center">
          <Link href="/">
            <div className="flex items-center gap-3 cursor-pointer">
              <img src={APP_LOGO} alt="iScore" className="h-12 w-auto" />
            </div>
          </Link>
          <Link href="/track">
            <Button variant="outline">تتبع الطلب</Button>
          </Link>
        </div>
      </header>

      <div className="container mx-auto px-4 py-12">
        <div className="max-w-2xl mx-auto">
          {isLoading ? (
            <Card>
              <CardContent className="py-12 text-center">
                <Loader2 className="h-12 w-12 animate-spin mx-auto text-cyan-600 mb-4" />
                <p>جاري التحميل...</p>
              </CardContent>
            </Card>
          ) : request ? (
            <Card>
              <CardHeader>
                <CardTitle className="text-2xl text-center">حالة الطلب</CardTitle>
                <CardDescription className="text-center">
                  رمز التحقق: {request.verificationCode}
                </CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="text-center py-8">
                  <div className="mb-4 flex justify-center">
                    {getStatusIcon(request.status)}
                  </div>
                  <h3 className="text-2xl font-bold mb-2">{getStatusText(request.status)}</h3>
                  
                  {request.status === "payment_submitted" && (
                    <div className="bg-cyan-50 p-6 rounded-lg mt-6">
                      <Loader2 className="h-8 w-8 animate-spin mx-auto text-cyan-600 mb-4" />
                      <p className="text-lg font-semibold mb-2">جاري التأكد من عملية الدفع</p>
                      <p className="text-sm text-gray-600">
                        سيتم مراجعة إثبات الدفع الخاص بك قريباً
                      </p>
                    </div>
                  )}

                  {request.status === "paid" && (
                    <div className="bg-green-50 p-6 rounded-lg mt-6">
                      <CheckCircle2 className="h-12 w-12 mx-auto text-green-600 mb-4" />
                      <p className="text-lg font-semibold mb-2">تم الدفع بنجاح</p>
                      <p className="text-sm text-gray-600">
                        جاري إرسال التقرير الائتماني
                      </p>
                    </div>
                  )}

                  {request.status === "completed" && request.reportPdfUrl && (
                    <div className="bg-green-50 p-6 rounded-lg mt-6 space-y-4">
                      <CheckCircle2 className="h-12 w-12 mx-auto text-green-600 mb-4" />
                      <p className="text-lg font-semibold mb-2">تم إعداد التقرير بنجاح!</p>
                      <p className="text-sm text-gray-600 mb-4">
                        يمكنك الآن تحميل التقرير الائتماني الخاص بك
                      </p>
                      
                      {/* PDF Preview */}
                      <div className="bg-white p-4 rounded-lg border">
                        <iframe
                          src={request.reportPdfUrl}
                          className="w-full h-96 rounded"
                          title="التقرير الائتماني"
                        />
                      </div>

                      <Button
                        onClick={() => request.reportPdfUrl && window.open(request.reportPdfUrl, "_blank")}
                        className="w-full"
                        size="lg"
                      >
                        <Download className="ml-2 h-5 w-5" />
                        تحميل التقرير
                      </Button>
                    </div>
                  )}
                </div>

                <div className="border-t pt-6">
                  <h4 className="font-bold mb-4">معلومات الطلب:</h4>
                  <div className="space-y-2 text-sm">
                    <div className="flex justify-between">
                      <span className="text-gray-600">اللغة:</span>
                      <span className="font-medium">{request.language === "ar" ? "العربية" : "الإنجليزية"}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">البريد الإلكتروني:</span>
                      <span className="font-medium">{request.email}</span>
                    </div>
                    <div className="flex justify-between">
                      <span className="text-gray-600">تاريخ الطلب:</span>
                      <span className="font-medium">
                        {new Date(request.createdAt).toLocaleDateString("ar-EG")}
                      </span>
                    </div>
                  </div>
                </div>

                <div className="flex gap-4">
                  <Link href="/" className="flex-1">
                    <Button variant="outline" className="w-full">
                      العودة للرئيسية
                    </Button>
                  </Link>
                  <Link href="/track" className="flex-1">
                    <Button className="w-full">
                      تتبع طلب آخر
                    </Button>
                  </Link>
                </div>
              </CardContent>
            </Card>
          ) : (
            <Card>
              <CardContent className="py-12 text-center">
                <p className="text-red-600">الطلب غير موجود</p>
              </CardContent>
            </Card>
          )}
        </div>
      </div>
      {request && <ChatWidget requestId={request.id} />}
    </div>
  );
}
