import { int, mysqlEnum, mysqlTable, text, timestamp, varchar, boolean } from "drizzle-orm/mysql-core";

/**
 * Core user table backing auth flow.
 */
export const users = mysqlTable("users", {
  id: int("id").autoincrement().primaryKey(),
  openId: varchar("openId", { length: 64 }).notNull().unique(),
  name: text("name"),
  email: varchar("email", { length: 320 }),
  loginMethod: varchar("loginMethod", { length: 64 }),
  role: mysqlEnum("role", ["user", "admin"]).default("user").notNull(),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
  lastSignedIn: timestamp("lastSignedIn").defaultNow().notNull(),
});

export type User = typeof users.$inferSelect;
export type InsertUser = typeof users.$inferInsert;

/**
 * طلبات التقرير الائتماني
 */
export const requests = mysqlTable("requests", {
  id: int("id").autoincrement().primaryKey(),
  language: varchar("language", { length: 10 }).notNull(), // ar or en
  idCardImageUrl: text("idCardImageUrl").notNull(),
  motherName: varchar("motherName", { length: 255 }).notNull(),
  idNumber: varchar("idNumber", { length: 50 }).notNull(),
  email: varchar("email", { length: 320 }).notNull(),
  phoneNumber: varchar("phoneNumber", { length: 50 }),
  paymentMethod: varchar("paymentMethod", { length: 100 }),
  paymentDetails: text("paymentDetails"),
  paymentProofUrl: text("paymentProofUrl"),
  status: mysqlEnum("status", ["pending_payment", "payment_submitted", "paid", "completed", "refunded"]).default("pending_payment").notNull(),
  verificationCode: varchar("verificationCode", { length: 20 }),
  smsVerificationCode: varchar("smsVerificationCode", { length: 6 }), // رمز التحقق عبر SMS
  smsVerified: int("smsVerified").default(0).notNull(), // 0 = لم يتم التحقق، 1 = تم التحقق
  reportPdfUrl: text("reportPdfUrl"),
  couponCode: varchar("couponCode", { length: 50 }), // كود الخصم المستخدم
  discountAmount: int("discountAmount").default(0).notNull(), // قيمة الخصم
  finalPrice: int("finalPrice").default(180).notNull(), // السعر النهائي بعد الخصم
  totalAmount: int("totalAmount").default(180).notNull(), // بالجنيه
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
});

export type Request = typeof requests.$inferSelect;
export type InsertRequest = typeof requests.$inferInsert;

/**
 * طرق الدفع المتاحة
 */
export const paymentMethods = mysqlTable("payment_methods", {
  id: int("id").autoincrement().primaryKey(),
  nameAr: varchar("nameAr", { length: 255 }).notNull(),
  nameEn: varchar("nameEn", { length: 255 }).notNull(),
  type: varchar("type", { length: 100 }).notNull(), // vodafone_cash, card, bank_transfer, etc.
  details: text("details").notNull(), // رقم الموبايل أو تفاصيل الحساب
  imageUrl: text("imageUrl"),
  isActive: boolean("isActive").default(true).notNull(),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
});

export type PaymentMethod = typeof paymentMethods.$inferSelect;
export type InsertPaymentMethod = typeof paymentMethods.$inferInsert;

/**
 * رسائل الدردشة
 */
export const messages = mysqlTable("messages", {
  id: int("id").autoincrement().primaryKey(),
  requestId: int("requestId").notNull(),
  senderType: mysqlEnum("senderType", ["user", "admin"]).notNull(),
  message: text("message").notNull(),
  isRead: boolean("isRead").default(false).notNull(),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
});

export type Message = typeof messages.$inferSelect;
export type InsertMessage = typeof messages.$inferInsert;

/**
 * Statistics table for storing manual stats
 */
export const statistics = mysqlTable("statistics", {
  id: int("id").autoincrement().primaryKey(),
  totalRequests: int("totalRequests").default(0).notNull(),
  totalRevenue: int("totalRevenue").default(0).notNull(), // in EGP
  totalUsers: int("totalUsers").default(0).notNull(),
  updatedAt: timestamp("updatedAt").defaultNow().onUpdateNow().notNull(),
});

export type Statistics = typeof statistics.$inferSelect;
export type InsertStatistics = typeof statistics.$inferInsert;

/**
 * Reviews table for service ratings
 */
export const reviews = mysqlTable("reviews", {
  id: int("id").autoincrement().primaryKey(),
  name: varchar("name", { length: 255 }).notNull(),
  rating: int("rating").notNull(), // 1-5
  comment: text("comment"),
  createdAt: timestamp("createdAt").defaultNow().notNull(),
});

export type Review = typeof reviews.$inferSelect;
export type InsertReview = typeof reviews.$inferInsert;

/**
 * جدول كوبونات الخصم
 */
export const coupons = mysqlTable("coupons", {
  id: int("id").autoincrement().primaryKey(),
  code: varchar("code", { length: 50 }).notNull().unique(),
  discountType: mysqlEnum("discount_type", ["percentage", "fixed"]).notNull(),
  discountValue: int("discount_value").notNull(), // نسبة مئوية أو مبلغ ثابت
  isActive: int("is_active").default(1).notNull(), // 1 = نشط، 0 = غير نشط
  usageLimit: int("usage_limit"), // عدد مرات الاستخدام المسموح (null = غير محدود)
  usedCount: int("used_count").default(0).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  expiresAt: timestamp("expires_at"),
});

export type Coupon = typeof coupons.$inferSelect;
export type InsertCoupon = typeof coupons.$inferInsert;
