/**
 * نظام إرسال الإشعارات عبر البريد الإلكتروني
 * يستخدم Manus Notification API لإرسال الإشعارات
 */

import { notifyOwner } from "./_core/notification";

export interface EmailNotification {
  to: string;
  subject: string;
  body: string;
  requestId?: number;
}

/**
 * إرسال إشعار بالبريد الإلكتروني للعميل
 * في الوقت الحالي، نرسل نسخة للمالك عبر نظام الإشعارات
 */
export async function sendEmailNotification(notification: EmailNotification): Promise<boolean> {
  const { to, subject, body, requestId } = notification;
  
  // إرسال إشعار للمالك عبر نظام الإشعارات المدمج
  const ownerNotification = `
📧 إشعار بريد إلكتروني جديد

إلى: ${to}
الموضوع: ${subject}
${requestId ? `رقم الطلب: #${requestId}` : ''}

---
${body}
  `;
  
  try {
    await notifyOwner({
      title: `إشعار: ${subject}`,
      content: ownerNotification,
    });
    
    console.log(`[Email] Notification sent to owner for: ${to}`);
    return true;
  } catch (error) {
    console.error(`[Email] Failed to send notification:`, error);
    return false;
  }
}

/**
 * إرسال إشعار عند إنشاء طلب جديد
 */
export async function sendRequestCreatedEmail(
  email: string,
  requestId: number,
  name: string
): Promise<boolean> {
  return sendEmailNotification({
    to: email,
    subject: "تم إنشاء طلبك بنجاح - iScore",
    requestId,
    body: `
مرحباً ${name}،

تم إنشاء طلبك للحصول على التقرير الائتماني بنجاح!

رقم الطلب: #${requestId}

يمكنك تتبع حالة طلبك من خلال الرابط التالي:
${process.env.VITE_FRONTEND_FORGE_API_URL || 'https://prosuitehub.com'}/payment-result/${requestId}

الخطوات التالية:
1. اختر طريقة الدفع المناسبة
2. قم برفع إثبات الدفع
3. سيتم مراجعة طلبك خلال 24 ساعة
4. ستحصل على تقريرك الائتماني عبر البريد الإلكتروني

شكراً لاختيارك iScore!
    `,
  });
}

/**
 * إرسال إشعار عند تغيير حالة الطلب
 */
export async function sendStatusChangeEmail(
  email: string,
  requestId: number,
  name: string,
  newStatus: string
): Promise<boolean> {
  const statusMessages: Record<string, { subject: string; body: string }> = {
    payment_submitted: {
      subject: "تم استلام إثبات الدفع - iScore",
      body: `
مرحباً ${name}،

تم استلام إثبات الدفع الخاص بك وجاري المراجعة.

رقم الطلب: #${requestId}

سيتم التحقق من الدفع خلال 24 ساعة وسنقوم بإشعارك فور الانتهاء.

شكراً لصبرك!
      `,
    },
    paid: {
      subject: "تم تأكيد الدفع - جاري إعداد التقرير - iScore",
      body: `
مرحباً ${name}،

تم تأكيد الدفع بنجاح! 🎉

رقم الطلب: #${requestId}

نحن الآن نعمل على إعداد تقريرك الائتماني. ستحصل على التقرير خلال 24-48 ساعة عبر البريد الإلكتروني.

شكراً لثقتك بنا!
      `,
    },
    completed: {
      subject: "تقريرك الائتماني جاهز! - iScore",
      body: `
مرحباً ${name}،

تقريرك الائتماني جاهز الآن! 📊

رقم الطلب: #${requestId}

يمكنك تحميل التقرير من خلال الرابط التالي:
${process.env.VITE_FRONTEND_FORGE_API_URL || 'https://prosuitehub.com'}/payment-result/${requestId}

نتمنى أن يكون التقرير مفيداً لك!
إذا كان لديك أي استفسارات، لا تتردد في التواصل معنا.

شكراً لاختيارك iScore!
      `,
    },
    refunded: {
      subject: "تم استرداد المبلغ - iScore",
      body: `
مرحباً ${name}،

تم استرداد المبلغ المدفوع بنجاح.

رقم الطلب: #${requestId}

إذا كان لديك أي استفسارات، يرجى التواصل معنا.

نعتذر عن أي إزعاج!
      `,
    },
  };

  const message = statusMessages[newStatus];
  if (!message) {
    console.warn(`[Email] No email template for status: ${newStatus}`);
    return false;
  }

  return sendEmailNotification({
    to: email,
    subject: message.subject,
    requestId,
    body: message.body,
  });
}
