import { COOKIE_NAME } from "@shared/const";
import { getSessionCookieOptions } from "./_core/cookies";
import { systemRouter } from "./_core/systemRouter";
import { publicProcedure, protectedProcedure, router } from "./_core/trpc";
import { z } from "zod";
import * as db from "./db";
import { sendRequestCreatedEmail, sendStatusChangeEmail } from "./email";;
import { storagePut } from "./storage";
import { TRPCError } from "@trpc/server";

// Helper to generate random verification code
function generateVerificationCode(): string {
  return Math.floor(100000 + Math.random() * 900000).toString();
}

export const appRouter = router({
  system: systemRouter,
  auth: router({
    me: publicProcedure.query(opts => opts.ctx.user),
    logout: publicProcedure.mutation(({ ctx }) => {
      const cookieOptions = getSessionCookieOptions(ctx.req);
      ctx.res.clearCookie(COOKIE_NAME, { ...cookieOptions, maxAge: -1 });
      return {
        success: true,
      } as const;
    }),
  }),

  // Request procedures
  request: router({
    create: publicProcedure
      .input(z.object({
        language: z.string(),
        motherName: z.string(),
        idNumber: z.string(),
        email: z.string(),
        phoneNumber: z.string().optional(),
      }))
      .mutation(async ({ input }) => {
        const verificationCode = generateVerificationCode();
        const requestId = await db.createRequest({
          ...input,
          idCardImageUrl: "", // Will be updated later
          verificationCode,
        });
        
        // إرسال إشعار بالبريد الإلكتروني
        await sendRequestCreatedEmail(input.email, requestId, input.motherName);
        
        return { success: true, verificationCode, requestId };
      }),

    updateIdCard: publicProcedure
      .input(z.object({
        requestId: z.number(),
        idCardImageUrl: z.string(),
      }))
      .mutation(async ({ input }) => {
        await db.updateRequest(input.requestId, {
          idCardImageUrl: input.idCardImageUrl,
        });
        return { success: true };
      }),

    updatePayment: publicProcedure
      .input(z.object({
        requestId: z.number(),
        paymentMethod: z.string(),
        paymentDetails: z.string(),
        paymentProofUrl: z.string(),
      }))
      .mutation(async ({ input }) => {
        await db.updateRequest(input.requestId, {
          paymentMethod: input.paymentMethod,
          paymentDetails: input.paymentDetails,
          paymentProofUrl: input.paymentProofUrl,
          status: "payment_submitted",
        });
        return { success: true };
      }),

    getById: publicProcedure
      .input(z.object({ id: z.number() }))
      .query(async ({ input }) => {
        const request = await db.getRequestById(input.id);
        if (!request) {
          throw new TRPCError({ code: "NOT_FOUND", message: "الطلب غير موجود" });
        }
        return request;
      }),

    getByVerificationCode: publicProcedure
      .input(z.object({ code: z.string() }))
      .query(async ({ input }) => {
        const request = await db.getRequestByVerificationCode(input.code);
        if (!request) {
          throw new TRPCError({ code: "NOT_FOUND", message: "الطلب غير موجود" });
        }
        return request;
      }),

    getByIdNumber: publicProcedure
      .input(z.object({ idNumber: z.string() }))
      .query(async ({ input }) => {
        const request = await db.getRequestByIdNumber(input.idNumber);
        if (!request) {
          throw new TRPCError({ code: "NOT_FOUND", message: "الطلب غير موجود" });
        }
        return request;
      }),

    getAll: protectedProcedure.query(async ({ ctx }) => {
      if (ctx.user.role !== "admin") {
        throw new TRPCError({ code: "FORBIDDEN" });
      }
      return await db.getAllRequests();
    }),

    updateStatus: protectedProcedure
      .input(z.object({
        requestId: z.number(),
        status: z.enum(["pending_payment", "payment_submitted", "paid", "completed", "refunded"]),
      }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.updateRequest(input.requestId, { status: input.status });
        
        // إرسال إشعار بالبريد الإلكتروني
        const request = await db.getRequestById(input.requestId);
        if (request) {
          await sendStatusChangeEmail(request.email, request.id, request.motherName, input.status);
        }
        
        return { success: true };
      }),

    uploadReport: protectedProcedure
      .input(z.object({
        requestId: z.number(),
        reportPdfUrl: z.string(),
      }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.updateRequest(input.requestId, {
          reportPdfUrl: input.reportPdfUrl,
          status: "completed",
        });
        return { success: true };
      }),
  }),

  // Payment method procedures
  paymentMethod: router({
    getActive: publicProcedure.query(async () => {
      return await db.getActivePaymentMethods();
    }),

    getAll: protectedProcedure.query(async ({ ctx }) => {
      if (ctx.user.role !== "admin") {
        throw new TRPCError({ code: "FORBIDDEN" });
      }
      return await db.getAllPaymentMethods();
    }),

    create: protectedProcedure
      .input(z.object({
        nameAr: z.string(),
        nameEn: z.string(),
        type: z.string(),
        details: z.string(),
        imageUrl: z.string().optional(),
      }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.createPaymentMethod(input);
        return { success: true };
      }),

    update: protectedProcedure
      .input(z.object({
        id: z.number(),
        nameAr: z.string().optional(),
        nameEn: z.string().optional(),
        type: z.string().optional(),
        details: z.string().optional(),
        imageUrl: z.string().optional(),
        isActive: z.boolean().optional(),
      }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        const { id, ...data } = input;
        await db.updatePaymentMethod(id, data);
        return { success: true };
      }),

    delete: protectedProcedure
      .input(z.object({ id: z.number() }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.deletePaymentMethod(input.id);
        return { success: true };
      }),
  }),

  // Message procedures
  message: router({
    send: publicProcedure
      .input(z.object({
        requestId: z.number(),
        message: z.string(),
      }))
      .mutation(async ({ input }) => {
        await db.createMessage({
          requestId: input.requestId,
          senderType: "user",
          message: input.message,
        });
        return { success: true };
      }),

    sendAdmin: protectedProcedure
      .input(z.object({
        requestId: z.number(),
        message: z.string(),
      }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.createMessage({
          requestId: input.requestId,
          senderType: "admin",
          message: input.message,
        });
        return { success: true };
      }),

    getByRequest: publicProcedure
      .input(z.object({ requestId: z.number() }))
      .query(async ({ input }) => {
        return await db.getMessagesByRequestId(input.requestId);
      }),

    markAsRead: protectedProcedure
      .input(z.object({ id: z.number() }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.markMessageAsRead(input.id);
        return { success: true };
      }),
  }),

  // Statistics procedures
  statistics: router({
    get: protectedProcedure.query(async ({ ctx }) => {
      if (ctx.user?.role !== "admin") {
        throw new TRPCError({ code: "FORBIDDEN" });
      }
      return await db.getStatistics();
    }),

    update: protectedProcedure
      .input(z.object({
        totalRequests: z.number().optional(),
        totalRevenue: z.number().optional(),
        totalUsers: z.number().optional(),
      }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user?.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.updateStatistics(input);
        return { success: true };
      }),
  }),

  // Review procedures
  review: router({
    create: publicProcedure
      .input(z.object({
        name: z.string(),
        rating: z.number().min(1).max(5),
        comment: z.string().optional(),
      }))
      .mutation(async ({ input }) => {
        await db.createReview(input);
        return { success: true };
      }),

    getAll: publicProcedure.query(async () => {
      return await db.getAllReviews();
    }),
  }),

  coupon: router({
    validate: publicProcedure
      .input(z.object({ code: z.string() }))
      .query(async ({ input }) => {
        return await db.validateCoupon(input.code);
      }),
    
    getAll: protectedProcedure
      .query(async ({ ctx }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        return await db.getAllCoupons();
      }),
    
    create: protectedProcedure
      .input(z.object({
        code: z.string(),
        discountType: z.enum(["percentage", "fixed"]),
        discountValue: z.number(),
        isActive: z.number().default(1),
        usageLimit: z.number().optional(),
        expiresAt: z.date().optional(),
      }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.createCoupon(input);
        return { success: true };
      }),
    
    update: protectedProcedure
      .input(z.object({
        id: z.number(),
        code: z.string().optional(),
        discountType: z.enum(["percentage", "fixed"]).optional(),
        discountValue: z.number().optional(),
        isActive: z.number().optional(),
        usageLimit: z.number().optional(),
        expiresAt: z.date().optional(),
      }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        const { id, ...updates } = input;
        await db.updateCoupon(id, updates);
        return { success: true };
      }),
    
    delete: protectedProcedure
      .input(z.object({ id: z.number() }))
      .mutation(async ({ ctx, input }) => {
        if (ctx.user.role !== "admin") {
          throw new TRPCError({ code: "FORBIDDEN" });
        }
        await db.deleteCoupon(input.id);
        return { success: true };
      }),
  }),
});

export type AppRouter = typeof appRouter;
